#include <windows.h>
#include <mmsystem.h>

#include <d3dx9.h>



typedef struct _DIRECT3D_STRUCT
{
   LPDIRECT3D9 pDirect3d9;
   LPDIRECT3DDEVICE9 pDirect3dDevice9;
   LPDIRECT3DVERTEXBUFFER9 pDirect3dVertexBuffers;
   LPDIRECT3DTEXTURE9 pDirect3dTexture;
   LPDIRECT3DINDEXBUFFER9 pDirect3dIndexBuffers;

} DIRECT3D_STRUCT, *PDIRECT3D_STRUCT;


typedef struct _CUSTOMVERTEX
{
    FLOAT x, y, z; 
    DWORD colour;
	FLOAT       tu, tv;
}CUSTOMVERTEX , *PCUSTOMVERTEX;

#define D3DFVF_CUSTOMVERTEX (D3DFVF_DIFFUSE | D3DFVF_XYZ | D3DFVF_TEX1)
/*
 * Prototypes
 */
BOOL Direct3d_SetupDeviceModes(PDIRECT3D_STRUCT pDirect3DStructure, HWND hWnd);
PDIRECT3D_STRUCT Direct3d_Initialize(HWND hWnd);
void Direct3d_UnInitialize(PDIRECT3D_STRUCT pDirect3DStructure);
HWND Example_CreateWindow(HINSTANCE hInstance);
LRESULT WINAPI Example_WindowProc(HWND hWnd, UINT uiMessage, WPARAM wParam, LPARAM lParam);
BOOL Direct3d_CreateCubeObject(PDIRECT3D_STRUCT pDirect3DStructure);
void Direct3D_Render(PDIRECT3D_STRUCT pDirect3DStructure);
void Direct3D_SetupTextureMap(PDIRECT3D_STRUCT pDirect3DStructure);
void Direct3D_SetupIndexes(PDIRECT3D_STRUCT pDirect3DStructure);

/************************************************************
 *
 * WinMain
 *
 *    This is the program entry point.
 *
 *
 ************************************************************/
int WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, PCHAR pszCmdLine, int iShow)
{
	HWND hWnd;
	PDIRECT3D_STRUCT pDirect3dStruct;
	MSG Msg;
	BOOL fMessage;

    hWnd = Example_CreateWindow(hInstance);

    pDirect3dStruct = Direct3d_Initialize(hWnd);

	if(pDirect3dStruct)
	{
		Direct3d_CreateCubeObject(pDirect3dStruct);
		Direct3D_SetupTextureMap(pDirect3dStruct);
		Direct3D_SetupIndexes(pDirect3dStruct);

		ShowWindow(hWnd, SW_SHOWDEFAULT);
        UpdateWindow(hWnd);

		PeekMessage(&Msg, NULL, 0U, 0U, PM_NOREMOVE);
	    
		while(Msg.message != WM_QUIT)
		{
			fMessage = PeekMessage(&Msg, NULL, 0U, 0U, PM_REMOVE);

			if(fMessage)
			{
				//Process message
				TranslateMessage(&Msg);
				DispatchMessage(&Msg);
			}
			else
			{
				Direct3D_Render(pDirect3dStruct);
				//No message to process, so render the current scene
				//Render();
			}

		}
		
	}

	return 0;
}

/************************************************************
 *
 * Direct3d_Initialize
 *
 *   Initializes Direct3D
 *
 ************************************************************/
PDIRECT3D_STRUCT Direct3d_Initialize(HWND hWnd)
{
	PDIRECT3D_STRUCT pDirect3D = NULL;
	BOOL bInitialized = FALSE;

	pDirect3D = (PDIRECT3D_STRUCT)LocalAlloc(LMEM_ZEROINIT, sizeof(DIRECT3D_STRUCT));

    if(pDirect3D)
    {
		pDirect3D->pDirect3d9 = Direct3DCreate9(D3D_SDK_VERSION);

		if(pDirect3D->pDirect3d9)
		{
			bInitialized = Direct3d_SetupDeviceModes(pDirect3D, hWnd);
		}
	}

	if(bInitialized == FALSE)
	{
		Direct3d_UnInitialize(pDirect3D);
		pDirect3D = NULL;
	}

	return pDirect3D;
}

/************************************************************
 *
 * Direct3d_UnInitialize
 *
 *   UnInit Direct3D
 *
 ************************************************************/
void Direct3d_UnInitialize(PDIRECT3D_STRUCT pDirect3DStructure)
{
	if(pDirect3DStructure)
	{
		LocalFree(pDirect3DStructure);
	}
}

/************************************************************
 *
 * Direct3d_CreateCubeObject
 *
 *   Creates a cube object
 *
 ************************************************************/
BOOL Direct3d_CreateCubeObject(PDIRECT3D_STRUCT pDirect3DStructure)
{
	BOOL bCubeCreated = FALSE;
	VOID* pVertices;
	/* 
	 * //Store each point of the cube together with it's colour
       //Make sure that the points of a polygon are specified in a clockwise direction,
       //this is because anti-clockwise faces will be culled
       //We will use a three triangle strips to render these polygons (Top, Sides, Bottom).
	 */
#if 0
    CUSTOMVERTEX cvVertices[] =
    { 
        //Top Face

		/*  

		*    *
		   *    *
        *    *
		   *   * 
  -5, 5, -5     5, 5, -5
  -5, 5, 5,     5, 5, 5

  -5, -5, -5    5, -5, -5
  -5, -5, 5     5, -5 ,5
 */
         // Top
         {-5.0f, 5.0f, -5.0f, D3DCOLOR_XRGB(255, 0, 0), 0.0, 0.0 },
         {-5.0f, 5.0f, 5.0f, D3DCOLOR_XRGB(255, 0, 0), 1.0, 0.0 },
		 {5.0f, 5.0f, -5.0f, D3DCOLOR_XRGB(255, 0, 0), 0.0, 1.0},
         {5.0f, 5.0f, 5.0f, D3DCOLOR_XRGB(255, 0, 0), 1.0, 1.0},
         
		 // Bottom
         {-5.0f, -5.0f, 5.0f, D3DCOLOR_XRGB(0, 255, 0), 0.0, 0.0},
         {-5.0f, -5.0f, -5.0f, D3DCOLOR_XRGB(0, 255, 0), 1.0, 0.0},
		 {5.0f, -5.0f, 5.0f, D3DCOLOR_XRGB(0, 255, 0), 0.0, 1.0},
         {5.0f, -5.0f, -5.0f, D3DCOLOR_XRGB(0, 255, 0), 1.0, 1.0},

		 // Front
         {-5.0f, -5.0f, -5.0f, D3DCOLOR_XRGB(255, 255, 255),0.0, 0.0},
         {-5.0f, 5.0f, -5.0f, D3DCOLOR_XRGB(255, 255, 255),1.0, 0.0},
		 {5.0f, -5.0f, -5.0f, D3DCOLOR_XRGB(255, 255, 255),0.0, 1.0},
         {5.0f, 5.0f, -5.0f, D3DCOLOR_XRGB(255, 255, 255),1.0, 1.0},

		 // Back
         {5.0f, -5.0f, 5.0f, D3DCOLOR_XRGB(255, 0, 0),0.0, 0.0},
         {5.0f, 5.0f, 5.0f, D3DCOLOR_XRGB(255, 0, 0),1.0, 0.0},
		 {-5.0f, -5.0f,5.0f, D3DCOLOR_XRGB(255, 0, 0),0.0, 1.0},
         {-5.0f, 5.0f, 5.0f, D3DCOLOR_XRGB(255, 0, 0),1.0, 1.0},

		 // Right Side
         {5.0f, -5.0f, -5.0f, D3DCOLOR_XRGB(0, 0, 255),0.0, 0.0},
         {5.0f, 5.0f, -5.0f, D3DCOLOR_XRGB(0, 0, 255),1.0, 0.0},
		 {5.0f, -5.0f, 5.0f, D3DCOLOR_XRGB(0, 0, 255),0.0, 1.0},
         {5.0f, 5.0f, 5.0f, D3DCOLOR_XRGB(0, 0, 255),1.0, 1.0},

     	 // Left Side 
         {-5.0f, -5.0f, 5.0f, D3DCOLOR_XRGB(50, 50, 0),0.0, 0.0},
         {-5.0f, 5.0f, 5.0f, D3DCOLOR_XRGB(50, 50, 0),1.0, 0.0},
		 {-5.0f, -5.0f, -5.0f, D3DCOLOR_XRGB(50, 50, 0),0.0, 1.0},
         {-5.0f, 5.0f, -5.0f, D3DCOLOR_XRGB(50, 50, 0),1.0, 1.0},


    /*       {-5.0f, 5.0f, -5.0f, D3DCOLOR_XRGB(255, 0, 0),}, //Vertex 0 - Blue 
        {-5.0f, 5.0f, 5.0f, D3DCOLOR_XRGB(255, 0, 0),}, //Vertex 1 - Red 
        {5.0f, 5.0f, -5.0f, D3DCOLOR_XRGB(255, 0, 0),}, //Vertex 2 - Red 
        {5.0f, 5.0f, 5.0f, D3DCOLOR_XRGB(255, 0, 0),}, //Vertex 3 - Green 

        //Face 1
        {-5.0f, -5.0f, -5.0f, D3DCOLOR_XRGB(0, 0, 255),}, //Vertex 4 - Red 
        {-5.0f, 5.0f, -5.0f, D3DCOLOR_XRGB(0, 0, 255),}, //Vertex 5 - Blue 
        {5.0f, -5.0f, -5.0f, D3DCOLOR_XRGB(0, 0, 255),}, //Vertex 6 - Green 
        {5.0f, 5.0f, -5.0f, D3DCOLOR_XRGB(0, 0, 255),}, //Vertex 7 - Red 

        //Face 2
        {5.0f, -5.0f, 5.0f, D3DCOLOR_XRGB(0, 255, 0),}, //Vertex 8 - Blue 
        {5.0f, 5.0f, 5.0f, D3DCOLOR_XRGB(0, 255, 0),}, //Vertex 9 - Green
        
        //Face 3
        {-5.0f, -5.0f, 5.0f, D3DCOLOR_XRGB(255, 0, 0),}, //Vertex 10 - Green 
        {-5.0f, 5.0f, 5.0f, D3DCOLOR_XRGB(255, 0, 0),}, //Vertex 11 - Red 

        //Face 4
        {-5.0f, -5.0f, -5.0f, D3DCOLOR_XRGB(0, 0, 255),}, //Vertex 12 - Red 
        {-5.0f, 5.0f, -5.0f, D3DCOLOR_XRGB(0, 0, 255),}, //Vertex 13 - Blue

        //Bottom Face
        {5.0f, -5.0f, -5.0f, D3DCOLOR_XRGB(0, 255, 0),}, //Vertex 14 - Green 
        {5.0f, -5.0f, 5.0f, D3DCOLOR_XRGB(0, 255, 0),}, //Vertex 15 - Blue 
        {-5.0f, -5.0f, -5.0f, D3DCOLOR_XRGB(0, 255, 0),}, //Vertex 16 - Red 
        {-5.0f, -5.0f, 5.0f, D3DCOLOR_XRGB(0, 255, 0),}, //Vertex 17 - Green
		*/
    };
#endif

	    CUSTOMVERTEX cvVertices[] =
    { 
        //Top Face

		/*  

		*    *
		   *    *
        *    *
		   *   * 
  -5, 5, -5     5, 5, -5
  -5, 5, 5,     5, 5, 5

  -5, -5, -5    5, -5, -5
  -5, -5, 5     5, -5 ,5
 */

         {-5.0f, 5.0f, -5.0f, D3DCOLOR_XRGB(255, 255, 255), 0.0, 0.0 },
         {-5.0f, 5.0f, 5.0f, D3DCOLOR_XRGB(255, 255, 255), 1.0, 0.0 },
		 {5.0f, 5.0f, -5.0f, D3DCOLOR_XRGB(255, 255, 255), 0.0, 1.0},
         {5.0f, 5.0f, 5.0f, D3DCOLOR_XRGB(255, 255, 255), 1.0, 1.0},
         
         {-5.0f, -5.0f, 5.0f, D3DCOLOR_XRGB(255, 255, 255), 0.0, 0.0},
         {-5.0f, -5.0f, -5.0f, D3DCOLOR_XRGB(255, 255, 255), 1.0, 0.0},
		 {5.0f, -5.0f, 5.0f, D3DCOLOR_XRGB(255, 255, 255), 0.0, 1.0},
         {5.0f, -5.0f, -5.0f, D3DCOLOR_XRGB(255, 255, 255), 1.0, 1.0},

         {5.0f, -5.0f, -5.0f, D3DCOLOR_XRGB(255, 255, 255), 0.0, 0.0},
         {5.0f, 5.0f, -5.0f, D3DCOLOR_XRGB(255, 255, 255), 1.0, 0.0 },
		 {5.0f, -5.0f, 5.0f, D3DCOLOR_XRGB(255, 255, 255), 0.0, 1.0},
		 {5.0f, 5.0f, 5.0f, D3DCOLOR_XRGB(255, 255, 255), 1.0, 1.0},

		 {-5.0f, -5.0f, 5.0f, D3DCOLOR_XRGB(255, 255, 255), 0.0, 0.0},
         {-5.0f, 5.0f, 5.0f, D3DCOLOR_XRGB(255, 255, 255), 1.0, 0.0 },
		 {-5.0f, -5.0f, -5.0f, D3DCOLOR_XRGB(255, 255, 255), 0.0, 1.0},
		 {-5.0f, 5.0f, -5.0f, D3DCOLOR_XRGB(255, 255, 255), 1.0, 1.0 },

    };
	HRESULT hr;

		 // 8, 9, 10,
		 // 11, 10, 9

    hr = pDirect3DStructure->pDirect3dDevice9->lpVtbl->CreateVertexBuffer(pDirect3DStructure->pDirect3dDevice9, sizeof(cvVertices), 0, D3DFVF_CUSTOMVERTEX, D3DPOOL_DEFAULT, &pDirect3DStructure->pDirect3dVertexBuffers, NULL);

	if(!FAILED(hr))
	{
		hr = pDirect3DStructure->pDirect3dVertexBuffers->lpVtbl->Lock(pDirect3DStructure->pDirect3dVertexBuffers, 0, sizeof(cvVertices), (BYTE**)&pVertices, 0);

		if(!FAILED(hr))
		{
			 memcpy(pVertices, cvVertices, sizeof(cvVertices));
			 pDirect3DStructure->pDirect3dVertexBuffers->lpVtbl->Unlock(pDirect3DStructure->pDirect3dVertexBuffers);
             
			 bCubeCreated = TRUE;
		}
	}

	return bCubeCreated;
}


/************************************************************
 *
 * Direct3d_SetupDeviceModes
 *
 *   Sets up the device modes
 *
 ************************************************************/
BOOL Direct3d_SetupDeviceModes(PDIRECT3D_STRUCT pDirect3DStructure, HWND hWnd)
{
	BOOL bSetupComplete = FALSE;
	HRESULT hResult;
	D3DDISPLAYMODE D3DDisplayMode = {0};
	D3DPRESENT_PARAMETERS D3DPresentParameters = {0};

    hResult = pDirect3DStructure->pDirect3d9->lpVtbl->GetAdapterDisplayMode(pDirect3DStructure->pDirect3d9, D3DADAPTER_DEFAULT, &D3DDisplayMode);

	if(!FAILED(hResult))
	{
		D3DPresentParameters.Windowed         = TRUE;
		D3DPresentParameters.SwapEffect       = D3DSWAPEFFECT_COPY;
        D3DPresentParameters.BackBufferFormat = D3DDisplayMode.Format;

		hResult = pDirect3DStructure->pDirect3d9->lpVtbl->CreateDevice(pDirect3DStructure->pDirect3d9, D3DADAPTER_DEFAULT, D3DDEVTYPE_HAL, hWnd, D3DCREATE_SOFTWARE_VERTEXPROCESSING, &D3DPresentParameters, &pDirect3DStructure->pDirect3dDevice9);

		if(!FAILED(hResult))
		{
		    pDirect3DStructure->pDirect3dDevice9->lpVtbl->SetRenderState(pDirect3DStructure->pDirect3dDevice9, D3DRS_CULLMODE, D3DCULL_CCW);
            pDirect3DStructure->pDirect3dDevice9->lpVtbl->SetRenderState(pDirect3DStructure->pDirect3dDevice9, D3DRS_LIGHTING, FALSE);

			bSetupComplete = TRUE;
		}		
	}

	return bSetupComplete;
}
		

/************************************************************
 *
 * Example_CreateWindow
 *
 *   Creates a Window
 *
 ************************************************************/
HWND Example_CreateWindow(HINSTANCE hInstance)
{
	HWND hWnd;
	WNDCLASSEX wc =    {sizeof(WNDCLASSEX), CS_CLASSDC, (WNDPROC)Example_WindowProc, 0L, 0L, 
                     hInstance, NULL, NULL, NULL, NULL, "DirectX Example", NULL};
	if(RegisterClassEx(&wc))
	{
         hWnd = CreateWindow("DirectX Example", "DirectX Example", WS_OVERLAPPEDWINDOW, 50, 50, 500, 500, GetDesktopWindow(), NULL, wc.hInstance, NULL);
	}

	return hWnd;
}




/************************************************************
 *
 * Example_WindowProc
 *
 *   Handles a Window
 *
 ************************************************************/
LRESULT WINAPI Example_WindowProc(HWND hWnd, UINT uiMessage, WPARAM wParam, LPARAM lParam)
{
	LRESULT lResult = 0;

	switch(uiMessage)
	{
	    case WM_CLOSE:
		case WM_DESTROY:
	    case WM_KEYUP:
       		 PostQuitMessage(0);
		     break;

	    default:
		lResult = DefWindowProc(hWnd, uiMessage, wParam, lParam);
	}


	return lResult;
}


void Direct3D_SetupRotation(PDIRECT3D_STRUCT pDirect3DStructure)
{
    //Here we will rotate our world around the x, y and z axis.
    D3DXMATRIX matWorld, matWorldX, matWorldY, matWorldZ;
	static float fRotation = 0;
	
	fRotation = timeGetTime()/900.0f;

    //Create the transformation matrices
    D3DXMatrixRotationX(&matWorldX, fRotation);
    D3DXMatrixRotationY(&matWorldY, fRotation);
    D3DXMatrixRotationZ(&matWorldZ, fRotation);

    //Combine the transformations by multiplying them together
    D3DXMatrixMultiply(&matWorld, &matWorldX, &matWorldY);
    D3DXMatrixMultiply(&matWorld, &matWorld, &matWorldZ);

    //Apply the tansformation
    pDirect3DStructure->pDirect3dDevice9->lpVtbl->SetTransform(pDirect3DStructure->pDirect3dDevice9, D3DTS_WORLD, &matWorld);
}

void Direct3D_SetupIndexes(PDIRECT3D_STRUCT pDirect3DStructure)
{
	UINT32 ciIndexes[] = {
          0, 1, 2,
		  3, 2, 1,

		  4, 5, 6, 
		  7, 6, 5,

		  5, 0, 7,
		  2, 7, 0,

		  6, 3, 4,
		  1, 4, 3,

		  8, 9, 10,
		  11, 10, 9,

		  12, 13, 14,
		  15, 14, 13
	} ;


	PVOID  pIndexes;
	HRESULT hr;

    hr = pDirect3DStructure->pDirect3dDevice9->lpVtbl->CreateIndexBuffer(pDirect3DStructure->pDirect3dDevice9, sizeof(ciIndexes), 0, D3DFMT_INDEX32, D3DPOOL_DEFAULT, &pDirect3DStructure->pDirect3dIndexBuffers, NULL);

	if(!FAILED(hr))
	{
		hr = pDirect3DStructure->pDirect3dIndexBuffers->lpVtbl->Lock(pDirect3DStructure->pDirect3dIndexBuffers, 0, sizeof(ciIndexes), (BYTE**)&pIndexes, 0);

		if(!FAILED(hr))
		{
			 memcpy(pIndexes, ciIndexes, sizeof(ciIndexes));
			 pDirect3DStructure->pDirect3dIndexBuffers->lpVtbl->Unlock(pDirect3DStructure->pDirect3dIndexBuffers);
             

		}
	}
}

void Direct3D_SetupCamera(PDIRECT3D_STRUCT pDirect3DStructure)
{
    //Here we will setup the camera.
    //The camera has three settings: "Camera Position", "Look at Position" and "Up Direction"
    //We have set the following:
    //Camera Position: (0, 0, -30)
    //Look at Position: (0, 0, 0)
    //Up direction: Y-Axis.
    D3DXMATRIX matView;
	D3DXVECTOR3 One = {0.0f, 0.0f,-30.0f}, Two = {0.0f, 0.0f, 0.0f}, Three = {0.0f, 1.0f, 0.0f};
    D3DXMatrixLookAtLH(&matView, &One, //Camera Position
                                 &Two, //Look At Position
                                 &Three); //Up Direction
    pDirect3DStructure->pDirect3dDevice9->lpVtbl->SetTransform(pDirect3DStructure->pDirect3dDevice9, D3DTS_VIEW, &matView);
}


void Direct3D_SetupTextureMap(PDIRECT3D_STRUCT pDirect3DStructure)
{
   HRESULT hResult;

   hResult = D3DXCreateTextureFromFile(pDirect3DStructure->pDirect3dDevice9, "image.bmp", &pDirect3DStructure->pDirect3dTexture);

   if(!FAILED(hResult))
   {
	   
   }
}

void Direct3D_SetupPerspective(PDIRECT3D_STRUCT pDirect3DStructure)
{
    //Here we specify the field of view, aspect ration and near and far clipping planes.
    D3DXMATRIX matProj;
    D3DXMatrixPerspectiveFovLH(&matProj, D3DX_PI/4, 1.0f, 1.0f, 500.0f);
    pDirect3DStructure->pDirect3dDevice9->lpVtbl->SetTransform(pDirect3DStructure->pDirect3dDevice9, D3DTS_PROJECTION, &matProj);
}


void Direct3D_Render(PDIRECT3D_STRUCT pDirect3DStructure)
{

    //Clear the backbuffer to black
    pDirect3DStructure->pDirect3dDevice9->lpVtbl->Clear(pDirect3DStructure->pDirect3dDevice9, 0, NULL, D3DCLEAR_TARGET, D3DCOLOR_XRGB(0, 0, 0), 1.0f, 0);
    
    //Begin the scene
    pDirect3DStructure->pDirect3dDevice9->lpVtbl->BeginScene(pDirect3DStructure->pDirect3dDevice9);
    
    //Setup the rotation, camera, and perspective matrices
    Direct3D_SetupRotation(pDirect3DStructure);
    Direct3D_SetupCamera(pDirect3DStructure);
    Direct3D_SetupPerspective(pDirect3DStructure);

	//Rendering our objects
    pDirect3DStructure->pDirect3dDevice9->lpVtbl->SetStreamSource(pDirect3DStructure->pDirect3dDevice9, 0, pDirect3DStructure->pDirect3dVertexBuffers, 0, sizeof(CUSTOMVERTEX));    //pDirect3DStructure->pDirect3dDevice9->lpVtbl->SetVertexShader(pDirect3DStructure->pDirect3dDevice9, );
	pDirect3DStructure->pDirect3dDevice9->lpVtbl->SetTexture(pDirect3DStructure->pDirect3dDevice9, 0, (IDirect3DBaseTexture9 *)pDirect3DStructure->pDirect3dTexture);
	pDirect3DStructure->pDirect3dDevice9->lpVtbl->SetIndices(pDirect3DStructure->pDirect3dDevice9, pDirect3DStructure->pDirect3dIndexBuffers);
    pDirect3DStructure->pDirect3dDevice9->lpVtbl->SetFVF(pDirect3DStructure->pDirect3dDevice9, D3DFVF_CUSTOMVERTEX);

	
    pDirect3DStructure->pDirect3dDevice9->lpVtbl->DrawIndexedPrimitive(pDirect3DStructure->pDirect3dDevice9, D3DPT_TRIANGLELIST, 0, 0, 16, 0, 12); 
#if 0
    pDirect3DStructure->pDirect3dDevice9->lpVtbl->DrawPrimitive(pDirect3DStructure->pDirect3dDevice9, D3DPT_TRIANGLESTRIP, 0, 2); 
    pDirect3DStructure->pDirect3dDevice9->lpVtbl->DrawPrimitive(pDirect3DStructure->pDirect3dDevice9, D3DPT_TRIANGLESTRIP, 4, 2); 
	pDirect3DStructure->pDirect3dDevice9->lpVtbl->DrawPrimitive(pDirect3DStructure->pDirect3dDevice9, D3DPT_TRIANGLESTRIP, 8, 2); 
	pDirect3DStructure->pDirect3dDevice9->lpVtbl->DrawPrimitive(pDirect3DStructure->pDirect3dDevice9, D3DPT_TRIANGLESTRIP, 12, 2); 
	pDirect3DStructure->pDirect3dDevice9->lpVtbl->DrawPrimitive(pDirect3DStructure->pDirect3dDevice9, D3DPT_TRIANGLESTRIP, 16, 2); 
	pDirect3DStructure->pDirect3dDevice9->lpVtbl->DrawPrimitive(pDirect3DStructure->pDirect3dDevice9, D3DPT_TRIANGLESTRIP, 20, 2); 
#endif     
    //End the scene
    pDirect3DStructure->pDirect3dDevice9->lpVtbl->EndScene(pDirect3DStructure->pDirect3dDevice9);
    
    //Filp the back and front buffers so that whatever has been rendered on the back buffer
    //will now be visible on screen (front buffer).
    pDirect3DStructure->pDirect3dDevice9->lpVtbl->Present(pDirect3DStructure->pDirect3dDevice9, NULL, NULL, NULL, NULL);
}

