/**********************************************************************
 * 
 *  Toby Opferman
 *
 *  Mutiny Driver!
 *
 *  This driver implements Operating System Mutiny!
 *  
 *  This example is for educational purposes only.  I license this source
 *  out for use in learning how to write a device driver.
 *
 *     Driver Entry Point  Copyright (c) 2005, All Rights Reserved
 **********************************************************************/



                                                                                                                                                  
#ifndef __PAGING64_H__
#define __PAGING64_H__

                                                                                                                               
/*
 *                                                                                                         +------------------------------------------------------------+
 *                                                             Page Directory Pointers      Page Directory \     Page Table Entries                                     \
 *                                                                                             Entries     \           ____________               Physical memory       \
 *  64-Bit Paging Overview                                                                                 \          |           |                                     \
 *                                                                                                         \         |           |                                      \
 *                                                                         +-------------------------------+        |           |                __________________     \
 *                                                                         \                                       |           |                \                 \     \
 *                          [       Linear Address                         \     ]                                |___________|                 \-----------------\     \
 *          ________       [ Sign Ext. |  PML4  | PDP  |  PDE | PTE  | Physical ]       +----------------------+                                \                 \     \
 *         |       |      63        48 47   39  38  30 29  21  20  12 11        0       \     ____________     \                                \-----------------\     \
 *        |       |                        \       \       \     \                      \    |           |     \      ____________              \                 \ <---+
 *       | GDT   |                         \       \       \     +----------------------+   |           |      \     |           |          +-->\-----------------\
 *      |       |                          \       \       \                               |           |       \    |           |           \   \                 \
 *     |_______|                           \       \       \                              |           |        +-> |           |------------+   \-----------------\
 *                                         \       \       +--------------------------+  |___________|            |           |                 \                 \
 *  The descriptor value                   \       \              ____________        \                      +-> |___________|                  \-----------------\
 *  is not calculated into                 \       \             |           |        \                      \                                  \                 \                 
 *  the paging unless                      \       \            |           |         \         ____________ \                                  \-----------------\
 *  compatibility mode                     \       \           |           |          \        |           | \                                  \                 \
 *  is specified.                          \       \          |           |           \       |           |  \        ____________              \-----------------\
 *                                         \       \         |___________|            \      |           |   \       |           |              \                 \
 *                                         \       \                                  \     |           |    \      |           |               \-----------------\
 *                                         \       \                                  \    |___________|     \     |           |                \                 \                 
 *                                         \       \            ____________          \                      \    |           |                 \-----------------\
 *                                         \       \           |           |          \                      \   |___________|                  \                 \
 *                            PML4         \       +--------->|           |---------+ \        ____________  \                                  \-----------------\
 *                     (Page Map Level 4)  \                 |           |          \ +----> |           |---+                                  \                 \
 *                                         \                |           |           \       |           |         ____________                  \-----------------\
 *                                         \       +------>|___________|            \      |           |         |           |                  \                 \                 
 *                                         \       \                                \     |           |         |           |                   \-----------------\
 *                            +------------+       \           ____________         +--->|___________|         |           |                    \                 \
 *                            \     PLM4           \         |           |                                    |           |                     \-----------------\
 *                            \   ____________     \        |           |                  ____________      |___________|                      \                 \
 *                            -->|           |-----+       |           |                  |           |                                         \_________________\
 *                              |           |             |           |                  |           |            ____________
 *                             |           |             |___________|                  |           |            |           |
 *                            |           |                                            |           |            |           | 
 *                     |---->|___________|                                            |___________|            |           |  
 *                    |                                                                                       |           |   
 *                   |                                                                                       |___________|    
 *        [         CR3             ]
 *
 *
 *
 *     The Linear Address that is in RIP or any other location accessing memory the high order bits
 *     are the "sign extend" bits.  This is an extension of bit 47 in the PML4 Index and according to
 *     the documentation it's for "canonical-address" forms.  I believe this basically just means
 *     you can savely add and subtract 64 bit addresses and a positive value can index into
 *     the same locations as a negative value.
 *
 *     CR3 points to a Page Map Level 4 table and the next high bits of the linear address index
 *     into this table.  This will then point to the base of a Page Directory Pointer table.  The
 *     next bits then index into this to get to the Page Directory Entry Table.  The next bits
 *     then index into this to get to finally a Page Table Entry.
 *       
 *     The Page Table entry then points to a 4k boundry of physical memory and the low order bits
 *     of the linear address index into this.
 *
 *     Now this is for 4k, there are also 2 Meg pages available which is then specified and indexed
 *     at the Page Directory Entries Table.  Then the rest of the low linear address, including
 *     what was originallly sepearted into PTE Indexing and Physical Indexing would be combined
 *     into Phyiscal Indexing.
 *
 */


/* Enable Write-Through Cache
 *  
 *    0 = Write Back caching policy
 *    1 = Write Through caching policy
 *          
 *  This determines the write cache policy for
 *  the paging table directly pointed to by CR3.
 *
 *  Write Through = Writes through to main memory. Write misses are
 *                  not allocated a new cache line.
 *
 *  Write Back =  Writes are updated on a "write back" instead of always
 *                going straight through to memory.  Write misses are
 *                allocated a new cache line.  This can provide
 *                better performance as you are not constantly writing
 *                straight through to main memory.
 */
#define CR3_FLAG_ENABLE_WRITE_THROUGH_CACHE64       0x8
#define CR3_FLAG_ENABLE_WRITE_THROUGH_CACHE_MASK64  0xF7

/* Disable Paging Cache
 *
 *   1 = Disable
 *   0 = Enable
 *
 *   This allows or disallows the paging table directly pointed
 *   to by CR3 to be cachable.
 *       
 */
#define CR3_FLAG_DISABLE_PAGE_CACHE64       0x10
#define CR3_FLAG_DISABLE_PAGE_CACHE_MASK64  0xEF
 

#pragma pack(1)
typedef struct _CR3_REGISTER64
{
    unsigned char  bFlags;
    unsigned short dwPhysicalAddressLowNibbleReserved;
    unsigned short wTopPhysicalAddress;
    unsigned char  bReserved;

} CR3_REGISTER64, *PCR3_REGISTER64;

/*
 *  Present Bit
 *
 *  Determines if the page pointed to by this
 *  entry is present in memory
 *
 */ 
#define PAGING64_FLAG_IS_PRESENT       0x01
#define PAGING64_FLAG_IS_PRESENT_MASK  0xFE

/*
 *  Writable Bit
 *
 *  Specifies if the page is writable.
 *
 */
#define PAGING64_FLAG_WRITABLE         0x02
#define PAGING64_FLAG_WRITABLE_MASK    0xFD

/*
 *  User/Supervisor Bit
 *
 *  1 = User Access is allowed
 *
 */
#define PAGING64_FLAG_USER             0x04
#define PAGING64_FLAG_USER_MASK        0xFB


/*
 *  Write Through Caching
 *
 *     Specifies if the page table pointed to
 *     by this entry is write through or write back
 *
 */
#define PAGING64_FLAG_WRITE_THROUGH_CACHE       0x08
#define PAGING64_FLAG_WRITE_THROUGH_CACHE_MASK  0xF7


/*
 *  Page Level Cache Disable
 *
 *     Specifies if the page tables pointed to
 *     by this entry are cachable.
 */
#define PAGING64_FLAG_CACHE_LEVEL_DISABLE       0x10
#define PAGING64_FLAG_CACHE_LEVEL_DISABLE_MASK  0xEF



/*
 *  Enable 2 Megabyte Pages (PDE)
 *
 *     1 = This PDE Points to physical memory and 2 Meg pages are used.
 *     0 = This PDE Points to a PTE and 4k pages are used.
 */
#define PAGING64_FLAG_ENABLE_2MEG_PAGES         0x80
#define PAGING64_FLAG_ENABLE_2MEG_PAGES_MASK    0x7F

/*  Page Atribute Bit (PTE/PDE (Lowest Paging Table)
 *
 *    This is an extension of the 
 *    PWT and PCD bits.  This is implemented differently
 *    on different processors but in AMD there is a
 *    register that is used.  This allows more granular
 *    caching attributes on the virtual address level.
 *
 */
#define PAGING64_FLAG_ATTRIBUTE_BIT       0x80
#define PAGING64_FLAG_ATTRIBUTE_BIT_MASK  0xEF

/*  Global Page Bit 
 *
 *    This allows pages to not be reloaded when
 *    CR3 is reloaded to help in caching.
 *  
 *    The only way to reload a "globally"
 *    cached page is through the INVLPG instruction.
 *
 */
#define PAGING64_FLAG_PAGE_IS_GLOBAL       0x00000001
#define PAGING64_FLAG_PAGE_IS_GLOBAL_MASK  0xFFFFFFFE



/*
 * 64 Bit Paging Data Structures
 *
 * 1. Page Map Level 4
 * 2. Page Directory Pointers
 * 3. Page Directory Entries
 * 4. Page Table Entries
 *
 *  64 bit Linear Address
 *  63 - 48   Sign Extension
 *  47 - 39   PML4 Entry Offset
 *  38 - 30   Page Directory Pointer Offset
 *  29 - 21   Page Directory Table Offset
 *  20 - 12   Page Table Offset
 *  11 - 0    Physical Page Offset
 */
#pragma pack(1)
typedef struct _PAGETABLE_PML4_ENTRY64
{
    unsigned char  bPagingAttributes;
    unsigned int   dwPhysicalAddressAndAvailableBits;
    unsigned short wPhysicalAddress;
    unsigned char  bNxBitAndPhysicalAddress;

} PAGETABLE_PML4_ENTRY64, *PPAGETABLE_PML4_ENTRY64;

/*
 * Attributes
 * 0 = P - Present Bit 
 * 1 = R/W - Read/Write Access
 * 2 = U/S - User/Supervisor (1 = User access is allowed)
 * 3 = PWT - Write Through/Cache Back Policy. 1 = WriteThrough
 * 4 = PCD - Page Level Cache Disable - If cacheable 
 * 5 = A - Accessed. Set by processor.
 * 6 = Ignored
 * 7 - 8 = MBZ - Reserved "MUST BE ZERO" field.
 * 9 - 11 = AVL - Not used by CPU, available to OS.
 * The other bits are Physical Address of PDP
 * NX Bit 63 - No Execute!  Requires an extended mode
 *             feature to be enabled.
 */
 

#pragma pack(1)
typedef struct _PAGETABLE_PDP_ENTRY64
{
    unsigned char  bPagingAttributes;
    unsigned int   dwPhysicalAddressAndAvailableBits;
    unsigned short wPhysicalAddress;
    unsigned char  bNxBitAndPhysicalAddress;

} PAGETABLE_PDP_ENTRY64, *PPAGETABLE_PDP_ENTRY64;

/* 
 * Attributes
 * 0 = P - Present Bit 
 * 1 = R/W - Read/Write Access
 * 2 = U/S - User/Supervisor (1 = User access is allowed)
 * 3 = PWT - Write Through/Cache Back Policy. 1 = WriteThrough
 * 4 = PCD - Page Level Cache Disable - If cacheable 
 * 5 = A - Accessed. Set by processor.
 * 6 = Ignored
 * 7 - 8 = MBZ - Reserved "MUST BE ZERO" field.
 * 9 - 11 = AVL - Not used by CPU, available to OS.
 * The other bits are Physical Address of PDP
 * NX Bit 63 - No Execute!  Requires an extended mode
 *             feature to be enabled.
 */



#pragma pack(1)
typedef struct _PAGETABLE_PDE_ENTRY64
{
    unsigned char  bPagingAttributes;
    unsigned int   dwPhysicalAddressAndAvailableBits;
    unsigned short wPhysicalAddress;
    unsigned char  bNxBitAndPhysicalAddress;

} PAGETABLE_PDE_ENTRY64, *PPAGETABLE_PDE_ENTRY64;

/* 
 * Attributes
 * 0 = P - Present Bit 
 * 1 = R/W - Read/Write Access
 * 2 = U/S - User/Supervisor (1 = User access is allowed)
 * 3 = PWT - Write Through/Cache Back Policy. 1 = WriteThrough
 * 4 = PCD - Page Level Cache Disable - If cacheable 
 * 5 = A - Accessed. Set by processor.
 * 6 = Ignored
 * 7 = PS bit, 0 = 4KB, 1 = 2 MEG pages. 
 * 8 = Ignored
 * 9 - 11 = AVL - Not used by CPU, available to OS.
 * The other bits are Physical Address of PDP
 * NX Bit 63 - No Execute!  Requires an extended mode
 *             feature to be enabled.
 */



/*
 * PTE's are only valid if 4KB Pages are enabled.  If 2 MB
 * pages are enabled the bits that would have index this
 * table (with the PDE) would now just index the higher
 * bits of the 2 MB page.
 * 
 * The PDE Entry defines the paging size
 */
#pragma pack(1)
typedef struct _PAGETABLE_PTE_ENTRY64
{
    unsigned char  bPagingAttributes;
    unsigned int   dwPhysicalAddressAndAvailableBits;
    unsigned short wPhysicalAddress;
    unsigned char  bNxBitAndPhysicalAddress;

} PAGETABLE_PTE_ENTRY64, *PPAGETABLE_PTE_ENTRY64;

/* 
 * Attributes
 * 0 = P - Present Bit 
 * 1 = R/W - Read/Write Access
 * 2 = U/S - User/Supervisor (1 = User access is allowed)
 * 3 = PWT - Write Through/Cache Back Policy. 1 = WriteThrough
 * 4 = PCD - Page Level Cache Disable - If cacheable 
 * 5 = A - Accessed. Set by processor.
 * 6 = D - Dirty.  Set by processor, page has been written to.
 * 7 = PAT - Page Attribute Bit
 * 8 - G - Global Page.
 * 9 - 11 = AVL - Not used by CPU, available to OS.
 * NX Bit 63 - No Execute.  Requires An extended mode feature
 *             to be enabled.
 */

NTSTATUS Paging64_CreatePageTables(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState);
NTSTATUS Paging64_MapVirtualToPhysical(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState, unsigned _int64  uiVirtualAddress64, unsigned _int64  uiPhysicalAddress64);
NTSTATUS Paging64_Map32BitPaeTo64BitPML4(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState);


#endif






