/**********************************************************************
 * 
 *  Toby Opferman
 *
 *  Mutiny Driver!
 *
 *  This driver implements Operating System Mutiny!
 *  
 *  This example is for educational purposes only.  I license this source
 *  out for use in learning how to write a device driver.
 *
 *     Driver Entry Point  Copyright (c) 2005, All Rights Reserved
 **********************************************************************/



                                                                                                                                                  
#ifndef __PAGING32_H__
#define __PAGING32_H__

                                                                                                            
/*
 *                                                                                         
 *                                                                  
 *                                                                                         
 *  32-Bit (NON-PAE) Paging Overview                                                       
 *                                                                                         
 *                                                                        
 *                                                                                    +-------------------------+
 *   [ Selector ]           [       Virtual Address    ]                              \     Physical Memory     \            
 *    \     ________         \                          +-----------------------------+    __________________   \
 *    \    |       |         \                          \                                 \                 \   \
 *    +-> |       |----------+--> [   Linear Address          ]                           \-----------------\   \
 *       | GDT   |               [ PDT   |   PTE  | Phsyical ]                            \                 \   \
 *      |       |               31    22  21   12 11        0                             \-----------------\   \
 *     |_______|                   \         \                                            \                 \   \
 *                                 \         \                                            \-----------------\<--+
 *  The descriptor value           \         \                                            \                 \
 *  the paging unless              \         \               Page Table        +--------> \-----------------\
 *  compatibility mode             \         \                 Entries         \          \                 \
 *  is specified.                  \         \                ____________     \          \-----------------\
 *                                 \         \               |           |     \          \                 \
 *                                 \         \              |           |      \          \-----------------\
 *                                 \         +------------>|           | ------+          \                 \                 
 *                                 \                      |           |                   \-----------------\
 *                                 \              +----> |___________|                    \                 \
 *                                 \              \                                       \-----------------\
 *                                 \              \                                       \                 \
 *                           +-----+              \         ____________                  \-----------------\
 *                           \                    \        |           |                  \                 \                 
 *                           \                    \       |           |                   \-----------------\
 *                           \  Page Directory    \      |           |                    \                 \
 *                           \     Table          \     |           |                     \-----------------\
 *                           \      ____________  \    |___________|                      \                 \
 *                           +---> |           |--+                                       \_________________\
 *                                |           |            ____________
 *                               |           |            |           |
 *                              |           |            |           | 
 *                     |------>|___________|            |           |  
 *                    |                                |           |   
 *                   |                                |___________|    
 *        [         CR3             ]
 *
 *
 *
 *     The Virtual Address in EIP is then indexed via the selector into the GDT.  The GDT is then
 *     adds it's base address onto the virtual address to make the linear address.  The linear
 *     address then finally indexes the page tables.
 */


/* Enable Write-Through Cache
 *  
 *    0 = Write Back caching policy
 *    1 = Write Through caching policy
 *          
 *  This determines the write cache policy for
 *  the paging table directly pointed to by CR3.
 *
 *  Write Through = Writes through to main memory. Write misses are
 *                  not allocated a new cache line.
 *
 *  Write Back =  Writes are updated on a "write back" instead of always
 *                going straight through to memory.  Write misses are
 *                allocated a new cache line.  This can provide
 *                better performance as you are not constantly writing
 *                straight through to main memory.
 */
#define CR3_FLAG_ENABLE_WRITE_THROUGH_CACHE32       0x8
#define CR3_FLAG_ENABLE_WRITE_THROUGH_CACHE_MASK32  0xF7

/* Disable Paging Cache
 *
 *   1 = Disable
 *   0 = Enable
 *
 *   This allows or disallows the paging table directly pointed
 *   to by CR3 to be cachable.
 *       
 */
#define CR3_FLAG_DISABLE_PAGE_CACHE32       0x10
#define CR3_FLAG_DISABLE_PAGE_CACHE_MASK32  0xEF
 


/* CR3 Register Layout
 *
 * Bits 0 2 Reserved
 * Bit 3 = Page Level Writethrough
 * Bit 4 = Page Level Cache Disable
 * Bit 5 - 11 Reserved
 * Bit 12 - 51 - Physical Address
 * Bit 52 - 63, Reserved
 */
#pragma pack(1)
typedef struct _CR3_REGISTER32
{
    unsigned char  bFlags;
    unsigned short wPhysicalAddressLowNibbleReserved;
    unsigned char  bTopPhysicalAddress;

} CR3_REGISTER32, *PCR3_REGISTER32;
/*
 *  Present Bit
 *
 *  Determines if the page pointed to by this
 *  entry is present in memory
 *
 */ 
#define PAGING_FLAG_IS_PRESENT32       0x01
#define PAGING_FLAG_IS_PRESENT_MASK32  0xFE

/*
 *  Writable Bit
 *
 *  Specifies if the page is writable.
 *
 */
#define PAGING_FLAG_WRITABLE32         0x02
#define PAGING_FLAG_WRITABLE_MASK32    0xFD

/*
 *  User/Supervisor Bit
 *
 *  1 = User Access is allowed
 *
 */
#define PAGING_FLAG_USER32             0x04
#define PAGING_FLAG_USER_MASK32        0xFB


/*
 *  Write Through Caching
 *
 *     Specifies if the page table pointed to
 *     by this entry is write through or write back
 *
 */
#define PAGING_FLAG_WRITE_THROUGH_CACHE32       0x08
#define PAGING_FLAG_WRITE_THROUGH_CACHE_MASK32  0xF7


/*
 *  Page Level Cache Disable
 *
 *     Specifies if the page tables pointed to
 *     by this entry are cachable.
 */
#define PAGING_FLAG_CACHE_LEVEL_DISABLE32       0x10
#define PAGING_FLAG_CACHE_LEVEL_DISABLE_MASK32  0xEF



/*
 *  Accessed Bit
 *
 *     1 = This page has been accessed
 *     0 = This page has not been accessed.
 */
#define PAGING_FLAG_ACCESS_BIT_MASK32    0xEF

/*  Dirty Bit (PTE Only)
 *
 *    1 = Page is dirty (has been modified)
 *    0 = Page is not dirty (has not been modified
 *
 */
#define PAGING_FLAG_DIRTY_BIT_MASK32     0xDF

/*  Global Page Bit 
 *
 *    This allows pages to not be reloaded when
 *    CR3 is reloaded to help in caching.
 *  
 *    The only way to reload a "globally"
 *    cached page is through the INVLPG instruction.
 *
 */
#define PAGING_FLAG_PAGE_IS_GLOBAL32       0x0001
#define PAGING_FLAG_PAGE_IS_GLOBAL_MASK32  0xFFFE


/*  Page Size Bit (PDE Only) 
 *
 *    0 = 4k Pages
 *    1 = 4 Megabyte Pages in regular mode or 2 Megabyte Pages in PAE Mode
 */
#define PAGING_FLAG_SET_2or4MEG_PAGES       0x80
#define PAGING_FLAG_SET_2or4MEG_PAGES_MASK  0x7E

#pragma pack(1)
typedef struct _PAGETABLE_PDE_ENTRY32
{
    unsigned char  bPagingAttributes;
    unsigned short wLowPhysicalAddress;
    unsigned char  wHighPhysicalAddress;

} PAGETABLE_PDE_ENTRY32, *PPAGETABLE_PDE_ENTRY32;

/* 
 * Attributes
 * 0 = P - Present Bit 
 * 1 = R/W - Read/Write Access
 * 2 = U/S - User/Supervisor (1 = User access is allowed)
 * 3 = PWT - Write Through/Cache Back Policy. 1 = WriteThrough
 * 4 = PCD - Page Level Cache Disable - If cacheable 
 * 5 = A - Accessed. Set by processor.
 * 6 = 0
 * 7 = PS bit, 0 = 4KB, 1 = 4 MEG pages or 2 Meg in PAE 
 * 8 = Global Page
 * 9 - 11 = AVL - Not used by CPU, available to OS.
 * The other bits are Physical Address of PDP
 */



/*
 * PTE's are only valid if 4KB Pages are enabled.  If 2 MB
 * pages are enabled the bits that would have index this
 * table (with the PDE) would now just index the higher
 * bits of the 2 MB page.
 * 
 * The PDE Entry defines the paging size
 */
#pragma pack(1)
typedef struct _PAGETABLE_PTE_ENTRY32
{
    unsigned char  bPagingAttributes;
    unsigned short wLowPhysicalAddress;
    unsigned char  wHighPhysicalAddress;

} PAGETABLE_PTE_ENTRY32, *PPAGETABLE_PTE_ENTRY32;

/* 
 * Attributes
 * 0 = P - Present Bit 
 * 1 = R/W - Read/Write Access
 * 2 = U/S - User/Supervisor (1 = User access is allowed)
 * 3 = PWT - Write Through/Cache Back Policy. 1 = WriteThrough
 * 4 = PCD - Page Level Cache Disable - If cacheable 
 * 5 = A - Accessed. Set by processor.
 * 6 = D - Dirty.  Set by processor, page has been written to.
 * 7 = 0
 * 8 - G - Global Page.
 * 9 - 11 = AVL - Not used by CPU, available to OS..
 */

NTSTATUS Paging32_CreatePageTables(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState);
NTSTATUS Paging32_DestroyPageTables(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState);

NTSTATUS Paging32_MapVirtualToPhysical(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState, unsigned int uiBaseAddress, unsigned int  uiVirtualAddress, unsigned int  uiPhysicalAddress);

#endif






