/**********************************************************************
 * 
 *  Toby Opferman
 *
 *  Mutiny Driver!
 *
 *  This driver implements Operating System Mutiny!
 *  
 *  This example is for educational purposes only.  I license this source
 *  out for use in learning how to write a device driver.
 *
 *     Driver Entry Point  Copyright (c) 2005, All Rights Reserved
 **********************************************************************/


#define _X86_

#include <ddk/ntddk.h>
#include "privatedata.h"

#define LINEAR_ADDRESS_TO_PDE_INDEX(x) ((x>>22) & 0x3FF)
#define LINEAR_ADDRESS_TO_PTE_INDEX(x) ((x>>12) & 0x3FF)

#define PHYSICAL_ADDRESS_TO_4K_BOUNDARY(h, l, a) \
               l = ((a>>8) & 0xFFF0);  \
               h = ((a>>24) & 0xFF);
               
#define PAGE_POOL_32BIT 31

                       
/*
 * Internal Functions
 */
PPAGETABLE_PTE_ENTRY32 Paging32_AllocatePageDirectoryEntry(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState, unsigned int uiPageDirectoryIndex);

/**********************************************************************
 * 
 *  Paging32_CreatePageTables
 *
 *    This API will Create a page table 
 *    in 32 bit
 *
 **********************************************************************/
NTSTATUS Paging32_CreatePageTables(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState)
{   
    NTSTATUS NtStatus = STATUS_SUCCESS;
     
    /*
     * We are going to pray that the Operating System will allocate our pages on 4k boundaries
     * so we don't have wasted bytes.  We need 1 Page for PDE, then 1 Page PER PTE unless we
     * do 4 Megabyte boundaries (which isn't a bad idea).
     *
     *
     */

    NtStatus = Memory_AllocateNonPaged4kPhysicallyAligned(PAGE_POOL_32BIT, &pOperatingSystemThunkState->PageTables32.OSAllocatedMemoryBlock);
                                       
    if(NT_SUCCESS(NtStatus))
    {
        PHYSICAL_ADDRESS_TO_4K_BOUNDARY(pOperatingSystemThunkState->PageTables32.CR3_32bit.bTopPhysicalAddress, pOperatingSystemThunkState->PageTables32.CR3_32bit.wPhysicalAddressLowNibbleReserved, pOperatingSystemThunkState->PageTables32.OSAllocatedMemoryBlock.Aligned4kPhysicalAddress);
            
        memset(pOperatingSystemThunkState->PageTables32.OSAllocatedMemoryBlock.pAllocatedVirtualMemory, 0, pOperatingSystemThunkState->PageTables32.OSAllocatedMemoryBlock.uiVirtualBlockSize);
        
        DbgPrint("32 bit Page Tables Created\n");
        DbgPrint("   Allocated Virtual Address %0x\n", pOperatingSystemThunkState->PageTables32.OSAllocatedMemoryBlock.pAllocatedVirtualMemory);
        DbgPrint("   Allocated Physical Address %0x\n", pOperatingSystemThunkState->PageTables32.OSAllocatedMemoryBlock.PhysicalAddress);
        DbgPrint("     Block Size %i\n", pOperatingSystemThunkState->PageTables32.OSAllocatedMemoryBlock.uiVirtualBlockSize);
        DbgPrint("   Adjusted Virtual Address %0x\n", pOperatingSystemThunkState->PageTables32.OSAllocatedMemoryBlock.pVirtualMappingOf4kPhysicalAddress);
        DbgPrint("   Adjusted Physical Address %0x\n", pOperatingSystemThunkState->PageTables32.OSAllocatedMemoryBlock.Aligned4kPhysicalAddress);
        DbgPrint("     Block Size %i\n", pOperatingSystemThunkState->PageTables32.OSAllocatedMemoryBlock.uiPageAlignedBlockSize);
        DbgPrint("  CR3 = %0x%0x00\n\n", pOperatingSystemThunkState->PageTables32.CR3_32bit.bTopPhysicalAddress, pOperatingSystemThunkState->PageTables32.CR3_32bit.wPhysicalAddressLowNibbleReserved);
    }

    return NtStatus;
}


/**********************************************************************
 * 
 *  Paging32_DestroyPageTables
 *
 *    This API will Destroy a page table 
 *    in 32 bit
 *
 **********************************************************************/
NTSTATUS Paging32_DestroyPageTables(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState)
{
    NTSTATUS NtStatus = STATUS_SUCCESS;

    Memory_FreeNonPaged4kAligned(&pOperatingSystemThunkState->PageTables32.OSAllocatedMemoryBlock);

    return NtStatus;
}




/**********************************************************************
 * 
 *  Paging32_MapVirtualToPhysical
 *
 *    This API will map a virtual address to a physical address
 *    in 32 bit
 *
 **********************************************************************/
NTSTATUS Paging32_MapVirtualToPhysical(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState, unsigned int uiBaseAddress, unsigned int  uiVirtualAddress, unsigned int  uiPhysicalAddress)
{   
    NTSTATUS NtStatus = STATUS_SUCCESS;
    unsigned int uiLinearAddress = uiBaseAddress + uiVirtualAddress;
    unsigned int uiPageDirectoryIndex = 0, uiPageTableIndex = 0;
    unsigned int uiAvailablePhysicalAddress = 0;
    BOOLEAN bFoundAvailablePhysicalPage = FALSE;
    PPAGETABLE_PTE_ENTRY32 pPageTableEntry;
    PHYSICAL_ADDRESS PhysicalAddress;
    unsigned short usHighAddress;
    unsigned short usLowAddress;

    uiPageDirectoryIndex = LINEAR_ADDRESS_TO_PDE_INDEX(uiLinearAddress);
    uiPageTableIndex     = LINEAR_ADDRESS_TO_PTE_INDEX(uiLinearAddress);

    DbgPrint("MapVirtualToPhysical\n");
    DbgPrint("   Base Descriptor Address = %0x\n", uiBaseAddress);
    DbgPrint("   Base Virtual Address = %0x\n", uiVirtualAddress);
    DbgPrint("   Adjusted Linear Address = %0x\n",uiLinearAddress);
    DbgPrint("   Desired Physical Address = %0x\n", uiPhysicalAddress);
    DbgPrint("   Page Directory Index = %0x, Page Table Index = %0x\n", uiPageDirectoryIndex, uiPageTableIndex);
    
    pPageTableEntry = Paging32_AllocatePageDirectoryEntry(pOperatingSystemThunkState, uiPageDirectoryIndex);

    if(pPageTableEntry)
    {

        if(pPageTableEntry[uiPageTableIndex].wHighPhysicalAddress == 0 && pPageTableEntry[uiPageTableIndex].wLowPhysicalAddress == 0)
        {
           pPageTableEntry[uiPageTableIndex].bPagingAttributes = PAGING_FLAG_IS_PRESENT32 | PAGING_FLAG_WRITABLE32;
           PHYSICAL_ADDRESS_TO_4K_BOUNDARY(pPageTableEntry[uiPageTableIndex].wHighPhysicalAddress, pPageTableEntry[uiPageTableIndex].wLowPhysicalAddress, uiPhysicalAddress);
        }
        else
        {
            DbgPrint("  Pages Already Mapped \n");
            PHYSICAL_ADDRESS_TO_4K_BOUNDARY(usHighAddress, usLowAddress, uiPhysicalAddress);
            
            if(pPageTableEntry[uiPageTableIndex].wHighPhysicalAddress != usHighAddress || pPageTableEntry[uiPageTableIndex].wLowPhysicalAddress != usLowAddress)
            {
                DbgPrint("  ********** Mapping Conflict!!! %0x%0x00 != %0x%0x00\n", pPageTableEntry[uiPageTableIndex].wHighPhysicalAddress, pPageTableEntry[uiPageTableIndex].wLowPhysicalAddress, usHighAddress, usLowAddress);
                _asm int 3
            }

            NtStatus = STATUS_UNSUCCESSFUL;
        }

        DbgPrint("   Index Page Table Virtual Address -> %0x\n", &pPageTableEntry[uiPageTableIndex]);
        PhysicalAddress = MmGetPhysicalAddress(&pPageTableEntry[uiPageTableIndex]);
        DbgPrint("   Index Page Table Physical Address -> %0x\n", (unsigned int)PhysicalAddress.QuadPart);
        DbgPrint("   Mapped To Physical Address %0x%0x00\n", pPageTableEntry[uiPageTableIndex].wHighPhysicalAddress, pPageTableEntry[uiPageTableIndex].wLowPhysicalAddress);
    }
    else
    {
        DbgPrint("No Page Tables\n");
        NtStatus = STATUS_UNSUCCESSFUL;
    }

    return NtStatus;
}

/**********************************************************************
 * 
 *  Paging32_AllocatePageDirectoryEntry
 *
 *    This API will Allocate a Page Directory Entry
 *
 **********************************************************************/
PPAGETABLE_PTE_ENTRY32 Paging32_AllocatePageDirectoryEntry(POPERATING_SYSTEM_THUNK_DATA pOperatingSystemThunkState, unsigned int uiPageDirectoryIndex)
{
    NTSTATUS NtStatus = STATUS_SUCCESS;
    PPAGETABLE_PDE_ENTRY32 pPageDirectoryEntry = (PPAGETABLE_PDE_ENTRY32)pOperatingSystemThunkState->PageTables32.OSAllocatedMemoryBlock.pVirtualMappingOf4kPhysicalAddress;
    PPAGETABLE_PTE_ENTRY32 pPageTableEntry = NULL;
    PHYSICAL_ADDRESS PhysicalAddress = {0};
    PVOID pPhysicalAdddressOfNewPageTable = 0;
    PVOID pVirtualAdddressOfNewPageTable = 0;
    
    if(pOperatingSystemThunkState->PageTables32.ptePageDirectoryTable[uiPageDirectoryIndex])
    {
       pPageTableEntry = pOperatingSystemThunkState->PageTables32.ptePageDirectoryTable[uiPageDirectoryIndex];
       
       DbgPrint("  Existing PDE Mapping (Index %0x) -> Page Table %0x\n", uiPageDirectoryIndex, pPageTableEntry);
    }
    else
    {
       pOperatingSystemThunkState->PageTables32.uiLastPageIndexAllocated++;

       pPhysicalAdddressOfNewPageTable = (PVOID)(pOperatingSystemThunkState->PageTables32.OSAllocatedMemoryBlock.Aligned4kPhysicalAddress + (pOperatingSystemThunkState->PageTables32.uiLastPageIndexAllocated<<12));
       pVirtualAdddressOfNewPageTable  = (PVOID)((unsigned char *)pOperatingSystemThunkState->PageTables32.OSAllocatedMemoryBlock.pVirtualMappingOf4kPhysicalAddress + (pOperatingSystemThunkState->PageTables32.uiLastPageIndexAllocated<<12));

       PHYSICAL_ADDRESS_TO_4K_BOUNDARY(pPageDirectoryEntry[uiPageDirectoryIndex].wHighPhysicalAddress, pPageDirectoryEntry[uiPageDirectoryIndex].wLowPhysicalAddress, (unsigned int)pPhysicalAdddressOfNewPageTable);
       pPageDirectoryEntry[uiPageDirectoryIndex].bPagingAttributes = PAGING_FLAG_IS_PRESENT32 | PAGING_FLAG_WRITABLE32;

       pPageTableEntry = pOperatingSystemThunkState->PageTables32.ptePageDirectoryTable[uiPageDirectoryIndex] = (PPAGETABLE_PTE_ENTRY32)pVirtualAdddressOfNewPageTable;

       /*
        * For Debugging Purposes
        */
       
       DbgPrint("  Creating new PDE Association PDE\n");
       DbgPrint("     PDE(Index %0x)\n", uiPageDirectoryIndex);
       DbgPrint("     Virtual Address Of PDE Entry(%0x)\n", &pPageDirectoryEntry[uiPageDirectoryIndex]);
       PhysicalAddress = MmGetPhysicalAddress(&pPageDirectoryEntry[uiPageDirectoryIndex]);
       DbgPrint("     Physical Address Of PDE Entry(%0x)\n", (unsigned int)PhysicalAddress.QuadPart);
       DbgPrint("     Virtual Address Of New Page Table(%0x)\n", pVirtualAdddressOfNewPageTable);
       PhysicalAddress = MmGetPhysicalAddress(pVirtualAdddressOfNewPageTable);
       DbgPrint("     Physical Address Of New Page Table(%0x), What we said it was (%0x)\n", (unsigned int)PhysicalAddress.QuadPart, pPhysicalAdddressOfNewPageTable);
       DbgPrint("     PDE Entry = (%0x%0x00)\n", pPageDirectoryEntry[uiPageDirectoryIndex].wHighPhysicalAddress, pPageDirectoryEntry[uiPageDirectoryIndex].wLowPhysicalAddress);
    }      
    
    return pPageTableEntry;
}


